/**
 * Manual MP4 Atom Parser for Cue Points
 * 
 * Pure JavaScript implementation that parses MP4 boxes directly
 * to extract chapter marker tracks (cue points).
 * 
 * This follows the MP4/ISO Base Media File Format specification
 * and reads chapter tracks created by mp4v2 library.
 */

const { readFileSync } = require('fs');
const logger = require('../logger');

/**
 * Parse XMP metadata to extract XSplit cue points
 * @param {string} xmpText - XMP XML text
 * @returns {Array} Array of cue point objects with {startTime, endTime, title}
 */
function parseXMPCuePoints(xmpText) {
  try {
    // Look for xsplit:broadcaster namespace
    if (!xmpText.includes('xsplit:broadcaster') && !xmpText.includes('xsplit')) {
      return [];
    }
    
    const cuePoints = [];
    
    // Extract the xsplit:broadcaster section
    const xsplitMatch = xmpText.match(/<xsplit:broadcaster[^>]*>([\s\S]*?)<\/xsplit:broadcaster>/i) ||
                       xmpText.match(/xmlns:xsplit[^>]*>([\s\S]*?)<\/[^>]*>/i);
    
    if (xsplitMatch && xsplitMatch[1]) {
      const xsplitSection = xsplitMatch[1];
      
      // Extract onCuepoint element - contains cue point timestamps
      const onCuepointMatch = xsplitSection.match(/<value\s+type="3"\s+name="onCuepoint"[^>]*>([\s\S]*?)<\/value>/i);
      
      if (onCuepointMatch && onCuepointMatch[1]) {
        const cuepointContent = onCuepointMatch[1];
        
        // Extract all timestamp values
        const timestampRegex = /<value\s+type="0"\s+name="timestamp"\s+value="([0-9]+)"\s*\/?>/gi;
        const timestamps = [];
        let match;
        
        while ((match = timestampRegex.exec(cuepointContent)) !== null) {
          const timestamp = parseInt(match[1], 10);
          if (!isNaN(timestamp)) {
            timestamps.push(timestamp);
          }
        }
        
        // Convert timestamps to cue points (timestamps are in milliseconds)
        for (let i = 0; i < timestamps.length; i++) {
          const timestampMs = timestamps[i];
          const startTime = timestampMs / 1000.0; // Convert to seconds
          const endTime = i < timestamps.length - 1 ? timestamps[i + 1] / 1000.0 : null;
          
          cuePoints.push({
            startTime: startTime,
            endTime: endTime,
            title: `Cue Point ${i + 1}`
          });
        }
        
        if (cuePoints.length > 0) {
          logger.log(`[CuePoints] Extracted ${cuePoints.length} cue points from XMP metadata`);
        }
      }
    }
    
    return cuePoints;
  } catch (error) {
    logger.error('[CuePoints] Error parsing XMP metadata:', error.message);
    return [];
  }
}

/**
 * Read a 32-bit unsigned integer (big-endian)
 */
function readUInt32BE(buffer, offset) {
  return buffer.readUInt32BE(offset);
}

/**
 * Read a 64-bit unsigned integer (big-endian)
 */
function readUInt64BE(buffer, offset) {
  const high = buffer.readUInt32BE(offset);
  const low = buffer.readUInt32BE(offset + 4);
  return (BigInt(high) << 32n) | BigInt(low);
}

/**
 * Read a 4-byte string (box type)
 */
function readBoxType(buffer, offset) {
  return buffer.toString('ascii', offset, offset + 4);
}

/**
 * Find a box in the MP4 file
 */
function findBox(buffer, boxType, startOffset = 0, endOffset = buffer.length) {
  let offset = startOffset;
  
  while (offset < endOffset - 8) {
    const size = readUInt32BE(buffer, offset);
    const type = readBoxType(buffer, offset + 4);
    
    if (type === boxType) {
      return { offset, size };
    }
    
    let boxSize = size;
    if (size === 1 && offset + 16 <= endOffset) {
      boxSize = Number(readUInt64BE(buffer, offset + 8));
      offset += boxSize;
    } else {
      offset += boxSize;
    }
    
    if (boxSize === 0) break;
  }
  
  return null;
}

/**
 * Read cue points from MP4 file using manual parsing
 */
async function readCuePointsWithParser(mp4Path) {
  try {
    const buffer = readFileSync(mp4Path);
    
    const moovBox = findBox(buffer, 'moov');
    if (!moovBox) {
      return [];
    }
    
    // Check for UUID boxes at file level (XSplit stores XMP metadata in UUID boxes)
    let fileOffset = 0;
    while (fileOffset < buffer.length - 8) {
      const size = readUInt32BE(buffer, fileOffset);
      if (size < 8) break;
      const type = readBoxType(buffer, fileOffset + 4);
      
      let boxSize = size;
      if (size === 1 && fileOffset + 16 <= buffer.length) {
        boxSize = Number(readUInt64BE(buffer, fileOffset + 8));
      }
      
      if (fileOffset === moovBox.offset) {
        fileOffset += boxSize;
        continue;
      }
      
      // Check for UUID boxes containing XMP metadata
      if (type === 'uuid' && boxSize > 16 && boxSize < 10000) {
        const dataStart = fileOffset + 24; // Skip size(4) + type(4) + uuid(16)
        const fullData = buffer.slice(dataStart, fileOffset + boxSize);
        
        try {
          const xmpText = fullData.toString('utf8', 0, fullData.length);
          if (xmpText.includes('xpacket') || xmpText.includes('xmpmeta') || xmpText.includes('xsplit')) {
            const cuePointsFromXMP = parseXMPCuePoints(xmpText);
            if (cuePointsFromXMP && cuePointsFromXMP.length > 0) {
              return cuePointsFromXMP;
            }
          }
        } catch (e) {
          // Not valid UTF-8, skip
        }
      }
      
      fileOffset += boxSize;
      if (boxSize === 0 || fileOffset >= buffer.length) break;
    }
    
    // No XMP metadata found in UUID boxes
    return [];
  } catch (error) {
    logger.error('[CuePoints] Error in manual parser:', error.message || error);
    return [];
  }
}

module.exports = {
  readCuePointsWithParser
};
